package com.anplus.hr.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.cola.dto.PageResponse;
import com.alibaba.cola.dto.Response;
import com.anplus.hr.config.EmployeeInfoCalcUtils;
import com.anplus.hr.constant.HrEmployeeConstants;
import com.anplus.hr.constant.HrFlowEnum;
import com.anplus.hr.constant.HrStatusEnum;
import com.anplus.hr.domain.EmployeeInfo;
import com.anplus.hr.domain.params.EmployeePartTimeInfoListParam;
import com.anplus.hr.domain.params.EmployeePartTimeInfoParam;
import com.anplus.hr.domain.vo.EmployeeInfoImportVo;
import com.anplus.hr.domain.vo.EmployeeInfoVo;
import com.anplus.hr.domain.vo.EmployeePartTimeInfoImportVo;
import com.anplus.hr.domain.vo.EmployeePartTimeInfoVo;
import com.anplus.hr.mapper.EmployeeInfoMapper;
import com.anplus.hr.service.EmployeeInfoServ;
import com.anplus.hr.service.EmployeePartTimeInfoServ;
import com.anplus.hr.service.EmployeeSysDeptServ;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import lombok.RequiredArgsConstructor;
import org.dromara.trans.service.impl.TransService;
import org.springframework.stereotype.Service;
import top.binfast.app.biz.sysapi.bean.model.oss.SysOss;
import top.binfast.common.core.enums.ResultCode;
import top.binfast.common.core.util.MapstructUtils;
import top.binfast.common.core.validate.AddGroup;
import top.binfast.common.excel.core.ExcelContextHolder;
import top.binfast.common.mybatis.util.QueryUtil;

import java.util.*;
import java.util.stream.Stream;

/**
 * 兼职员工信息Service业务层处理
 *
 * @author LiuBin
 * @date 2025-12-10
 */
@RequiredArgsConstructor
@Service
public class EmployeePartTimeInfoServImpl extends ServiceImpl<EmployeeInfoMapper, EmployeeInfo> implements EmployeePartTimeInfoServ {

    private final EmployeeInfoMapper employeeInfoMapper;
    private final EmployeeInfoServ employeeInfoServ;
    private final EmployeeSysDeptServ employeeSysDeptServ;
    private final TransService transService;

    /**
     * 分页查询兼职员工信息列表
     *
     * @param param 查询条件
     * @return 兼职员工信息分页列表
     */
    @Override
    public PageResponse<EmployeePartTimeInfoVo> queryPageList(EmployeePartTimeInfoListParam param) {
        Page<EmployeeInfo> page = QueryUtil.getPage(param);
        LambdaQueryWrapper<EmployeeInfo> lambdaQuery = this.buildQueryWrapper(param);
        employeeInfoMapper.selectPage(page, lambdaQuery);
        return QueryUtil.getPageResponse(page, MapstructUtils.convert(page.getRecords(), EmployeePartTimeInfoVo.class));
    }


    /**
     * 查询符合条件的兼职员工信息列表
     *
     * @param param 查询条件
     * @return 兼职员工信息列表
     */
    @Override
    public List<EmployeePartTimeInfoVo> queryList(EmployeePartTimeInfoListParam param) {
        LambdaQueryWrapper<EmployeeInfo> lambdaQuery = this.buildQueryWrapper(param);
        return MapstructUtils.convert(employeeInfoMapper.selectList(lambdaQuery), EmployeePartTimeInfoVo.class);
    }

    private LambdaQueryWrapper<EmployeeInfo> buildQueryWrapper(EmployeePartTimeInfoListParam param) {
        LambdaQueryWrapper<EmployeeInfo> lambdaQuery = Wrappers.<EmployeeInfo>lambdaQuery();
        lambdaQuery.orderByDesc(EmployeeInfo::getId);
        lambdaQuery.eq(EmployeeInfo::getPartTimeFlag, HrEmployeeConstants.YES);
//        lambdaQuery.eq(StrUtil.isNotBlank(param.getPlate()), EmployeeInfo::getPlate, param.getPlate());
//        lambdaQuery.eq(StrUtil.isNotBlank(param.getFirstLevelDepartment()), EmployeeInfo::getFirstLevelDepartment, param.getFirstLevelDepartment());
//        lambdaQuery.eq(StrUtil.isNotBlank(param.getSecondLevelDepartment()), EmployeeInfo::getSecondLevelDepartment, param.getSecondLevelDepartment());
//        lambdaQuery.eq(StrUtil.isNotBlank(param.getThirdLevelDepartment()), EmployeeInfo::getThirdLevelDepartment, param.getThirdLevelDepartment());
//        lambdaQuery.eq(param.getDeptId() != null, EmployeeInfo::getDeptId, param.getDeptId());
        if (ObjectUtil.isNotNull(param.getDeptId())) {
            //优先单部门搜索
            lambdaQuery.eq(EmployeeInfo::getDeptId, param.getDeptId());
        } else if (ObjectUtil.isNotNull(param.getBelongDeptId())) {
            //部门树搜索
            lambdaQuery.and(x -> {
                x.in(EmployeeInfo::getDeptId, employeeSysDeptServ.buildQueryWrapper(param.getBelongDeptId()));
            });
        }
        lambdaQuery.eq(StrUtil.isNotBlank(param.getPositionType()), EmployeeInfo::getPositionType, param.getPositionType());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getPosition()), EmployeeInfo::getPosition, param.getPosition());
        lambdaQuery.like(StrUtil.isNotBlank(param.getName()), EmployeeInfo::getName, param.getName());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getGender()), EmployeeInfo::getGender, param.getGender());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getIdCardNumber()), EmployeeInfo::getIdCardNumber, param.getIdCardNumber());
        lambdaQuery.eq(param.getOssId() != null, EmployeeInfo::getOssId, param.getOssId());
        lambdaQuery.eq(param.getBirthDate() != null, EmployeeInfo::getBirthDate, param.getBirthDate());
        lambdaQuery.eq(param.getAge() != null, EmployeeInfo::getAge, param.getAge());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getNativePlace()), EmployeeInfo::getNativePlace, param.getNativePlace());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getEthnicity()), EmployeeInfo::getEthnicity, param.getEthnicity());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getMaritalStatus()), EmployeeInfo::getMaritalStatus, param.getMaritalStatus());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getPoliticalStatus()), EmployeeInfo::getPoliticalStatus, param.getPoliticalStatus());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getPhoneNumber()), EmployeeInfo::getPhoneNumber, param.getPhoneNumber());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getEmergencyContact()), EmployeeInfo::getEmergencyContact, param.getEmergencyContact());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getEmergencyContactPhone()), EmployeeInfo::getEmergencyContactPhone, param.getEmergencyContactPhone());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getHomeAddress()), EmployeeInfo::getHomeAddress, param.getHomeAddress());
        lambdaQuery.eq(param.getEntryDate() != null, EmployeeInfo::getEntryDate, param.getEntryDate());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getFulltimeEducation()), EmployeeInfo::getFulltimeEducation, param.getFulltimeEducation());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getFulltimeSchool()), EmployeeInfo::getFulltimeSchool, param.getFulltimeSchool());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getFulltimeMajor()), EmployeeInfo::getFulltimeMajor, param.getFulltimeMajor());
        lambdaQuery.eq(param.getFulltimeGraduationDate() != null, EmployeeInfo::getFulltimeGraduationDate, param.getFulltimeGraduationDate());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getFulltimeDegree()), EmployeeInfo::getFulltimeDegree, param.getFulltimeDegree());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getEmploymentForm()), EmployeeInfo::getEmploymentForm, param.getEmploymentForm());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getPartTimeFlag()), EmployeeInfo::getPartTimeFlag, param.getPartTimeFlag());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getPartTimeAgreement()), EmployeeInfo::getPartTimeAgreement, param.getPartTimeAgreement());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getAgreementEntity()), EmployeeInfo::getAgreementEntity, param.getAgreementEntity());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getAgreementPeriod()), EmployeeInfo::getAgreementPeriod, param.getAgreementPeriod());
        lambdaQuery.eq(param.getAgreementStartDate() != null, EmployeeInfo::getAgreementStartDate, param.getAgreementStartDate());
        lambdaQuery.eq(param.getAgreementEndDate() != null, EmployeeInfo::getAgreementEndDate, param.getAgreementEndDate());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getRemarks()), EmployeeInfo::getRemarks, param.getRemarks());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getBankCardNumber()), EmployeeInfo::getBankCardNumber, param.getBankCardNumber());
        lambdaQuery.like(StrUtil.isNotBlank(param.getBankName()), EmployeeInfo::getBankName, param.getBankName());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getHasRelativeInCompany()), EmployeeInfo::getHasRelativeInCompany, param.getHasRelativeInCompany());
        lambdaQuery.like(StrUtil.isNotBlank(param.getRelativeName()), EmployeeInfo::getRelativeName, param.getRelativeName());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getIntroducer()), EmployeeInfo::getIntroducer, param.getIntroducer());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getIntroducerRelation()), EmployeeInfo::getIntroducerRelation, param.getIntroducerRelation());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getSalaryLocation()), EmployeeInfo::getSalaryLocation, param.getSalaryLocation());
        lambdaQuery.like(StrUtil.isNotBlank(param.getCostOfDept()), EmployeeInfo::getCostOfDept, param.getCostOfDept());
        return lambdaQuery;
    }

    @Override
    public Response importPartTimeEmployeeList(Stream<EmployeePartTimeInfoImportVo> list) {
        List<EmployeePartTimeInfoImportVo> errorList = new ArrayList<>();
        List<EmployeePartTimeInfoImportVo> successList = new ArrayList<>();
        Set<String> idCardNumberSet = new HashSet<>();
        list.forEach(item -> {
            item.validGroup(AddGroup.class);
            if (item.hasError()) {
                errorList.add(item);
                return;
            }
            if (employeeInfoServ.checkEmployeeIdCardNumberUnique(item.getIdCardNumber(), null) ||
                    idCardNumberSet.contains(item.getIdCardNumber())) {
                item.addError("身份证号已存在");
                errorList.add(item);
                return;
            }
            idCardNumberSet.add(item.getIdCardNumber());
            successList.add(item);
        });
        if (CollUtil.isNotEmpty(successList)) {
            Map<String, Long> deptNamesIdMap = employeeSysDeptServ.selectJoinDeptNames();
            List<EmployeeInfo> insertList = new ArrayList<>(successList.size());
            for (EmployeePartTimeInfoImportVo importVo : successList) {
                EmployeeInfoImportVo deptInfo = MapstructUtils.convert(importVo, EmployeeInfoImportVo.class);
                Long leafDeptId = deptNamesIdMap.get(employeeInfoServ.buildDeptNameStr(deptInfo));
                if (leafDeptId == null) {
                    importVo.addError("部门不存在");
                    errorList.add(importVo);
                    continue;
                }
                EmployeeInfo employeeInfo = MapstructUtils.convert(importVo, EmployeeInfo.class);
                employeeInfo.setDeptId(leafDeptId);
                employeeInfo.setStatus(employeeInfo.getEntryDate() == null
                        ? HrStatusEnum.DRAFT.getStatus() : HrStatusEnum.ENTRY.getStatus());
                employeeInfo.setPartTimeFlag(HrEmployeeConstants.YES);
                EmployeeInfoCalcUtils.calcAgeGroup(employeeInfo);
//                EmployeeInfoCalcUtils.calcExpectedRegularDate(employeeInfo);
                insertList.add(employeeInfo);
            }
            employeeInfoMapper.insert(insertList);
        }
        StringBuilder message;
        if (CollUtil.isNotEmpty(errorList)) {
            ExcelContextHolder.setErrorExist();
            message = new StringBuilder("共" + errorList.size() + "条数据导入失败，错误如下：<br/>");
            errorList.forEach(item -> message.append(item.defaultFailMsg()));
            return Response.buildFailure(ResultCode.FAIL.getCode(), message.toString());
        } else {
            message = new StringBuilder("共" + successList.size() + "条数据导入成功");
            Response response = Response.buildSuccess();
            response.setErrMessage(message.toString());
            return response;
        }
    }

    /**
     * 查询兼职员工信息
     *
     * @param id 主键
     * @return 兼职员工信息
     */
    @Override
    public EmployeePartTimeInfoVo queryById(Long id) {
        EmployeeInfo employeePartTimeInfo = employeeInfoMapper.selectById(id);
        return MapstructUtils.convert(employeePartTimeInfo, EmployeePartTimeInfoVo.class);
    }

    @Override
    public EmployeePartTimeInfoVo infoDetail(Long id) {
        EmployeeInfo employeeInfo = employeeInfoMapper.selectById(id);
        EmployeePartTimeInfoVo employeeInfoVo = MapstructUtils.convert(employeeInfo, EmployeePartTimeInfoVo.class);
        transService.transOne(employeeInfoVo);
        return employeeInfoVo;
    }

    /**
     * 新增兼职员工信息
     *
     * @param param 兼职员工信息
     * @return 是否新增成功
     */
    @Override
    public Boolean insertByParam(EmployeePartTimeInfoParam param) {
        EmployeeInfo employeePartTimeInfo = MapstructUtils.convert(param, EmployeeInfo.class);
        // 更新员工年龄组
        EmployeeInfoCalcUtils.calcAgeGroup(employeePartTimeInfo);
        employeePartTimeInfo.setPartTimeFlag(HrEmployeeConstants.YES);
//        employeePartTimeInfo.setEntryApplyStatus(HrFlowEnum.DRAFT.getStatus());
        employeePartTimeInfo.setStatus(HrStatusEnum.DRAFT.getStatus());
        return this.save(employeePartTimeInfo);
    }

    /**
     * 修改兼职员工信息
     *
     * @param param 兼职员工信息
     * @return 是否修改成功
     */
    @Override
    public Boolean updateByParam(EmployeePartTimeInfoParam param) {
        EmployeeInfo employeePartTimeInfo = MapstructUtils.convert(param, EmployeeInfo.class);
        return this.updateById(employeePartTimeInfo);
    }

    /**
     * 保存前的数据校验
     */
    private void validEntityBeforeSave(EmployeeInfo entity) {
        // 做一些数据校验,如唯一约束
    }

    /**
     * 校验并批量删除兼职员工信息信息
     *
     * @param ids 待删除的主键集合
     * @return 是否删除成功
     */
    @Override
//    @Transactional(rollbackFor = {Exception.class})
    public Boolean delByIds(List<Long> ids) {
        //做一些业务上的校验,判断是否需要校验
        return this.removeByIds(ids);
    }
}
