package com.anplus.hr.service.impl;

import cn.hutool.core.convert.Convert;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.cola.dto.PageResponse;
import com.anplus.hr.constant.EmployeeChangeLogTypeConstant;
import com.anplus.hr.constant.HrResignYearsOfServiceTypeEnum;
import com.anplus.hr.constant.HrResignationTypeConstant;
import com.anplus.hr.domain.EmployeeChangeLog;
import com.anplus.hr.domain.EmployeeInfo;
import com.anplus.hr.domain.params.EmployeeChangeLogListParam;
import com.anplus.hr.domain.params.EmployeeChangeLogParam;
import com.anplus.hr.domain.vo.EmployeeChangeLogVo;
import com.anplus.hr.mapper.EmployeeChangeLogMapper;
import com.anplus.hr.service.EmployeeChangeLogServ;
import com.anplus.hr.service.EmployeeSysDeptServ;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Service;
import top.binfast.app.biz.sysapi.bean.vo.sysDept.SysDeptDTO;
import top.binfast.app.biz.sysbiz.service.SysDeptServ;
import top.binfast.common.core.util.MapstructUtils;
import top.binfast.common.mybatis.util.QueryUtil;

import java.time.LocalDate;
import java.util.List;

/**
 * 员工异动记录Service业务层处理
 *
 * @author LiuBin
 * @date 2025-11-17
 */
@RequiredArgsConstructor
@Service
public class EmployeeChangeLogServImpl extends ServiceImpl<EmployeeChangeLogMapper, EmployeeChangeLog> implements EmployeeChangeLogServ {

    private final EmployeeChangeLogMapper employeeChangeLogMapper;
    private final SysDeptServ sysDeptServ;
    private final EmployeeSysDeptServ employeeSysDeptServ;

    /**
     * 分页查询员工异动记录列表
     *
     * @param param 查询条件
     * @return 员工异动记录分页列表
     */
    @Override
    public PageResponse<EmployeeChangeLogVo> queryPageList(EmployeeChangeLogListParam param) {
        Page<EmployeeChangeLog> page = QueryUtil.getPage(param);
        LambdaQueryWrapper<EmployeeChangeLog> lambdaQuery = this.buildQueryWrapper(param);
        employeeChangeLogMapper.selectPage(page, lambdaQuery);
        return QueryUtil.getPageResponse(page, MapstructUtils.convert(page.getRecords(), EmployeeChangeLogVo.class));
    }


    /**
     * 查询符合条件的员工异动记录列表
     *
     * @param param 查询条件
     * @return 员工异动记录列表
     */
    @Override
    public List<EmployeeChangeLogVo> queryList(EmployeeChangeLogListParam param) {
        LambdaQueryWrapper<EmployeeChangeLog> lambdaQuery = this.buildQueryWrapper(param);
        return MapstructUtils.convert(employeeChangeLogMapper.selectList(lambdaQuery), EmployeeChangeLogVo.class);
    }

    private LambdaQueryWrapper<EmployeeChangeLog> buildQueryWrapper(EmployeeChangeLogListParam param) {
        LambdaQueryWrapper<EmployeeChangeLog> lambdaQuery = Wrappers.<EmployeeChangeLog>lambdaQuery();
        lambdaQuery.orderByDesc(EmployeeChangeLog::getId);
        lambdaQuery.eq(StrUtil.isNotBlank(param.getType()), EmployeeChangeLog::getType, param.getType());
//        lambdaQuery.eq(param.getDeptId() != null, EmployeeChangeLog::getDeptId, param.getDeptId());
        if (ObjectUtil.isNotNull(param.getDeptId())) {
            //优先单部门搜索
            lambdaQuery.eq(EmployeeChangeLog::getDeptId, param.getDeptId());
        } else if (ObjectUtil.isNotNull(param.getBelongDeptId())) {
            //部门树搜索
            lambdaQuery.and(x -> {
                x.in(EmployeeChangeLog::getDeptId, employeeSysDeptServ.buildQueryWrapper(param.getBelongDeptId()));
            });
        }
        lambdaQuery.eq(param.getEmployeeId() != null, EmployeeChangeLog::getEmployeeId, param.getEmployeeId());
        lambdaQuery.eq(param.getChangeDate() != null, EmployeeChangeLog::getChangeDate, param.getChangeDate());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getResignType()), EmployeeChangeLog::getResignType, param.getResignType());
        lambdaQuery.eq(StrUtil.isNotBlank(param.getResignYearsOfServiceType()), EmployeeChangeLog::getResignYearsOfServiceType, param.getResignYearsOfServiceType());
        return lambdaQuery;
    }

    /**
     * 查询员工异动记录
     *
     * @param id 主键
     * @return 员工异动记录
     */
    @Override
    public EmployeeChangeLogVo queryById(Long id) {
        EmployeeChangeLog employeeChangeLog = employeeChangeLogMapper.selectById(id);
        return MapstructUtils.convert(employeeChangeLog, EmployeeChangeLogVo.class);
    }

    @Override
    public Boolean insertByEmployee(String applyCode, EmployeeInfo employeeInfo, LocalDate changeDate, String changeType) {
        SysDeptDTO childDept = sysDeptServ.selectDeptById(employeeInfo.getDeptId());
        List<String> split = StrUtil.splitTrim(childDept.getNodePath(), '/');
        if (split.size() < 2) {
            return true;
        }
        Long parentDeptId = split.stream().map(Convert::toLong).filter(t -> t > 0).findFirst().get();
        SysDeptDTO parentDept = sysDeptServ.selectDeptById(parentDeptId);
//        List<EmployeeChangeLog> changeLogList = new ArrayList<>();
//        for (String changeType : changeTypes) {
        EmployeeChangeLog changeLog = new EmployeeChangeLog();
        changeLog.setApplyCode(applyCode);
        changeLog.setEmployeeId(employeeInfo.getId());
        changeLog.setDeptId(employeeInfo.getDeptId());
        changeLog.setPlateId(parentDeptId);
        changeLog.setPlate(parentDept.getName());
        changeLog.setPosition(employeeInfo.getPosition());
        changeLog.setType(changeType);
        changeLog.setChangeDate(changeDate != null ? changeDate : LocalDate.now());
        if (EmployeeChangeLogTypeConstant.Resign.equals(changeType)) {
            changeLog.setResignType(HrResignationTypeConstant.FIRED.equals(employeeInfo.getResignationType()) ?
                    HrResignationTypeConstant.FIRED : HrResignationTypeConstant.HAND_IN_WORK);
            HrResignYearsOfServiceTypeEnum hrResignYearsOfServiceTypeEnum;
            if (HrResignationTypeConstant.PROBATION_NOT_PASSED.equals(employeeInfo.getResignationType())) {
                hrResignYearsOfServiceTypeEnum = HrResignYearsOfServiceTypeEnum.PROBATION_PERIOD;
            } else {
                hrResignYearsOfServiceTypeEnum = HrResignYearsOfServiceTypeEnum.getByTotalMonths(employeeInfo.getYearsOfServiceMonths());
            }
            if (hrResignYearsOfServiceTypeEnum != null) {
                changeLog.setResignYearsOfServiceType(hrResignYearsOfServiceTypeEnum.getCode());
            }
        }
//            changeLogList.add(changeLog);
//        }
        return this.save(changeLog);
    }

    /**
     * 新增员工异动记录
     *
     * @param param 员工异动记录
     * @return 是否新增成功
     */
    @Override
    public Boolean insertByParam(EmployeeChangeLogParam param) {
        EmployeeChangeLog employeeChangeLog = MapstructUtils.convert(param, EmployeeChangeLog.class);
        return this.save(employeeChangeLog);
    }

    /**
     * 修改员工异动记录
     *
     * @param param 员工异动记录
     * @return 是否修改成功
     */
    @Override
    public Boolean updateByParam(EmployeeChangeLogParam param) {
        EmployeeChangeLog employeeChangeLog = MapstructUtils.convert(param, EmployeeChangeLog.class);
        return this.updateById(employeeChangeLog);
    }

    /**
     * 保存前的数据校验
     */
    private void validEntityBeforeSave(EmployeeChangeLog entity) {
        // 做一些数据校验,如唯一约束
    }

    /**
     * 校验并批量删除员工异动记录信息
     *
     * @param ids 待删除的主键集合
     * @return 是否删除成功
     */
    @Override
//    @Transactional(rollbackFor = {Exception.class})
    public Boolean delByIds(List<Long> ids) {
        //做一些业务上的校验,判断是否需要校验
        return this.removeByIds(ids);
    }
}
