package com.anplus.hr.controller;

import cn.dev33.satoken.annotation.SaCheckPermission;
import com.alibaba.cola.dto.PageResponse;
import com.alibaba.cola.dto.Response;
import com.alibaba.cola.dto.SingleResponse;
import com.anplus.hr.domain.params.EmployeeTrainingListParam;
import com.anplus.hr.domain.params.EmployeeTrainingParam;
import com.anplus.hr.domain.vo.EmployeeTrainingImportVo;
import com.anplus.hr.domain.vo.EmployeeTrainingVo;
import com.anplus.hr.service.EmployeeTrainingServ;
import jakarta.annotation.Resource;
import jakarta.validation.constraints.Min;
import jakarta.validation.constraints.NotEmpty;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import top.binfast.common.core.constant.BusinessType;
import top.binfast.common.core.util.ResponseUtils;
import top.binfast.common.core.validate.AddGroup;
import top.binfast.common.core.validate.EditGroup;
import top.binfast.common.excel.annotion.ExcelExport;
import top.binfast.common.excel.annotion.ExcelStream;
import top.binfast.common.excel.core.ExcelDownHandler;
import top.binfast.common.log.annotation.PinSysLog;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Stream;

/**
 * 员工培训档案
 *
 * @author LiuBin
 * @date 2025-12-07
 */
@Validated
@RestController
@RequestMapping("/employee/training")
public class EmployeeTrainingCtrl {

    @Resource
    private EmployeeTrainingServ employeeTrainingServ;

    /**
     * 查询员工培训档案列表
     */
    @SaCheckPermission("employee:training:list")
    @GetMapping("/page")
    public PageResponse<EmployeeTrainingVo> pageList(EmployeeTrainingListParam param) {
        return employeeTrainingServ.queryPageList(param);
    }

    /**
     * 导出员工培训档案列表
     */
    @ExcelExport
    @SaCheckPermission("employee:training:export")
    @PinSysLog(value = "员工培训档案", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public List<EmployeeTrainingVo> export(EmployeeTrainingListParam param) {
        return employeeTrainingServ.queryList(param);
    }

    /**
     * 导入数据
     */
    @PinSysLog(value = "员工培训档案", businessType = BusinessType.IMPORT)
    @SaCheckPermission("employee:training:import")
    @PostMapping(value = "/import")
    public Response importData(@ExcelStream Stream<EmployeeTrainingImportVo> stream) {
        return employeeTrainingServ.importTrainingList(stream);
    }

    /**
     * 获取导入模板
     */
    @ExcelExport(clazz = EmployeeTrainingImportVo.class, writeHandler = ExcelDownHandler.class)
    @PostMapping("/importTemplate")
    public List<EmployeeTrainingImportVo> importTemplate() {
        return new ArrayList<>();
    }

    /**
     * 获取员工培训档案详细信息
     *
     * @param id 主键
     */
    @SaCheckPermission("employee:training:query")
    @GetMapping("/{id}")
    public SingleResponse<EmployeeTrainingVo> getDetail(@PathVariable @Min(1)
                                                        Long id) {
        return SingleResponse.of(employeeTrainingServ.queryById(id));
    }

    /**
     * 新增员工培训档案
     */
    @SaCheckPermission("employee:training:add")
    @PinSysLog(value = "员工培训档案", businessType = BusinessType.INSERT)
    @PostMapping()
    public Response add(@Validated(AddGroup.class) @RequestBody EmployeeTrainingParam param) {
        return ResponseUtils.ofResult(employeeTrainingServ.insertByParam(param));
    }

    /**
     * 修改员工培训档案
     */
    @SaCheckPermission("employee:training:edit")
    @PinSysLog(value = "员工培训档案", businessType = BusinessType.UPDATE)
    @PutMapping()
    public Response edit(@Validated(EditGroup.class) @RequestBody EmployeeTrainingParam param) {
        return ResponseUtils.ofResult(employeeTrainingServ.updateByParam(param));
    }

    /**
     * 删除员工培训档案
     *
     * @param ids 主键串
     */
    @SaCheckPermission("employee:training:remove")
    @PinSysLog(value = "员工培训档案", businessType = BusinessType.DELETE)
    @DeleteMapping("/{ids}")
    public Response remove(@NotEmpty(message = "主键不能为空")
                           @PathVariable Long[] ids) {
        return ResponseUtils.ofResult(employeeTrainingServ.delByIds(List.of(ids)));
    }
}